<?php
session_start();
require_once 'models/User.php';
require_once 'includes/remember-me.php';
require_once 'config/google-oauth.php';

// Clear any existing session for testing
if (isset($_GET['clear'])) {
    session_destroy();
    clearRememberMe();
    header('Location: login.php');
    exit();
}

// Check for remember me token and auto-login
if (checkRememberMe()) {
    // Redirect to main page
    header('Location: index.php');
    exit();
}

// Only redirect if already logged in and not a guest
if (isset($_SESSION['user_id']) && $_SESSION['user_id'] !== 'guest' && !isset($_GET['debug'])) {
    header('Location: index.php');
    exit();
}

$error_message = '';
$success_message = '';

// Handle OAuth errors
if (isset($_GET['error'])) {
    switch ($_GET['error']) {
        case 'oauth_cancelled':
            $error_message = 'Google sign-in was cancelled.';
            break;
        case 'oauth_failed':
            $error_message = 'Google sign-in failed. Please try again.';
            if (isset($_GET['debug'])) {
                $error_message .= ' Debug: ' . htmlspecialchars($_GET['debug']);
            }
            break;
    }
}

// Handle login success
if (isset($_GET['login']) && $_GET['login'] === 'success') {
    $success_message = 'Successfully signed in with Google!';
}

// Handle login form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = trim($_POST['email'] ?? '');
    $password = trim($_POST['password'] ?? '');
    $remember_me = isset($_POST['remember_me']) && $_POST['remember_me'] === '1';
    
    if (empty($email) || empty($password)) {
        $error_message = 'Please fill in all fields.';
    } else {
        $user = new User();
        $user_data = $user->verifyPassword($email, $password);
        
        if ($user_data) {
            $_SESSION['user_id'] = $user_data['id'];
            $_SESSION['user_email'] = $user_data['email'];
            $_SESSION['user_name'] = $user_data['first_name'] . ' ' . $user_data['last_name'];
            
            // Handle remember me functionality
            if ($remember_me) {
                $remember_token = $user->createRememberToken($user_data['id'], true);
                if ($remember_token) {
                    // Set cookie for 30 days
                    setcookie('remember_token', $remember_token, time() + (30 * 24 * 60 * 60), '/', '', false, true);
                }
            }
            
            $success_message = 'Login successful! Redirecting...';
            // Redirect after 2 seconds to show success message
            echo '<script>setTimeout(function(){ window.location.href = "index.php"; }, 2000);</script>';
        } else {
            $error_message = 'Invalid email or password.';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - 2 Days Parfume</title>
    <link rel="stylesheet" href="styles.css">
    <link rel="stylesheet" href="auth.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="script.js" defer></script>
</head>
<body>
    <div class="auth-container">
        <!-- Top Navigation -->
        <nav class="top-nav">
            <div class="nav-left">
                <i class="fas fa-arrow-left" onclick="window.location.href='index.php'"></i>
            </div>
            <div class="nav-center">
                <div class="logo">
                    <img src="images/LOGO.png" alt="2 Days Parfume Logo" class="logo-img">
                </div>
            </div>
            <div class="nav-right">
                <i class="fas fa-search" onclick="window.location.href='products.php'"></i>
            </div>
        </nav>

        <!-- Auth Content -->
        <div class="auth-content">
            <div class="auth-header">
                <h1 class="auth-title">Welcome Back!</h1>
                <p class="auth-subtitle">Sign in to your account.</p>
            </div>

            <!-- Error/Success Messages -->
            <?php if ($error_message): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i>
                    <?php echo htmlspecialchars($error_message); ?>
                </div>
            <?php endif; ?>

            <?php if ($success_message): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i>
                    <?php echo htmlspecialchars($success_message); ?>
                </div>
            <?php endif; ?>

            <!-- Login Form -->
            <form class="auth-form" method="POST" action="">
                <div class="form-group">
                    <input type="email" name="email" id="email" class="form-input" placeholder="Email" required>
                </div>
                
                <div class="form-group">
                    <input type="password" name="password" id="password" class="form-input" placeholder="Password" required>
                </div>

                <div class="form-options">
                    <label class="remember-me">
                        <input type="checkbox" name="remember_me" id="remember_me" value="1">
                        <span class="checkmark"></span>
                        Remember Me
                    </label>
                    <a href="#" class="forgot-password">Forgot Password?</a>
                </div>

                <button type="submit" class="auth-btn primary">
                    <span>Login</span>
                </button>
            </form>

            <!-- Social Login -->
            <!-- <div class="social-divider">
                <span>Or Continue With</span>
            </div>

            <div class="social-buttons">
                <button class="social-btn google" onclick="signInWithGoogle()">
                    <i class="fab fa-google"></i>
                    <span>Continue with Google</span>
                </button>
            </div> -->


            <!-- Sign Up Link -->
            <div class="auth-footer">
                <p>Not A Member? <a href="signup.php" class="auth-link">Join Now</a></p>
            </div>
        </div>
    </div>

    <script>
        function signInWithGoogle() {
            // Mobile-specific OAuth handling
            const isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
            const oauthUrl = '<?php echo getGoogleOAuthURL(); ?>';
            
            if (isMobile) {
                // For mobile, use a more reliable method
                try {
                    // Try opening in same window first
                    window.location.replace(oauthUrl);
                } catch (e) {
                    // Fallback to regular redirect
                    window.location.href = oauthUrl;
                }
            } else {
                // Desktop behavior
                window.location.href = oauthUrl;
            }
        }
    </script>
</body>
</html>