<?php
require_once __DIR__ . '/../config/database.php';

class Order {
    private $conn;
    private $table_name = "orders";
    private $order_items_table = "order_items";

    public function __construct() {
        $this->conn = getDB();
    }

    /**
     * Check if sufficient stock is available for order items
     */
    public function checkStockAvailability($items) {
        // Group items by product_id and sum quantities
        $stock_requirements = [];
        foreach ($items as $item) {
            $product_id = $item['product_id'];
            if (isset($stock_requirements[$product_id])) {
                $stock_requirements[$product_id] += $item['quantity'];
            } else {
                $stock_requirements[$product_id] = $item['quantity'];
            }
        }
        
        // Check stock for each unique product
        foreach ($stock_requirements as $product_id => $total_quantity) {
            $stock_query = "SELECT stock_quantity FROM products WHERE id = :product_id";
            $stock_stmt = $this->conn->prepare($stock_query);
            $stock_stmt->bindValue(':product_id', $product_id, PDO::PARAM_INT);
            $stock_stmt->execute();
            $product = $stock_stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$product) {
                error_log("Product not found for stock check: " . $product_id);
                return false;
            }
            
            if ($product['stock_quantity'] < $total_quantity) {
                error_log("Insufficient stock for product " . $product_id . ": Required " . $total_quantity . ", Available " . $product['stock_quantity']);
                return false;
            }
        }
        return true;
    }

    /**
     * Create a new order
     */
    public function create($order_data) {
        try {
            // Validate required data
            if (empty($order_data['first_name']) || empty($order_data['last_name']) || 
                empty($order_data['email']) || empty($order_data['phone']) ||
                empty($order_data['address']) || empty($order_data['city']) ||
                empty($order_data['state']) || empty($order_data['zip_code']) ||
                empty($order_data['payment_method'])) {
                error_log("Order creation error: Missing required fields");
                return false;
            }
            
            if (empty($order_data['items']) || !is_array($order_data['items'])) {
                error_log("Order creation error: No items in cart");
                return false;
            }
            
            // Check stock availability before creating order
            if (!$this->checkStockAvailability($order_data['items'])) {
                error_log("Order creation error: Insufficient stock");
                return false;
            }
            
            $this->conn->beginTransaction();
            
            // Generate order number
            $order_number = 'ORD' . date('Ymd') . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
            
            // Prepare shipping address
            $shipping_address = json_encode([
                'first_name' => $order_data['first_name'],
                'last_name' => $order_data['last_name'],
                'email' => $order_data['email'],
                'phone' => $order_data['phone'],
                'address' => $order_data['address'],
                'city' => $order_data['city'],
                'state' => $order_data['state'],
                'zip_code' => $order_data['zip_code'],
                'country' => $order_data['country']
            ]);
            
            // Insert order
            $query = "INSERT INTO " . $this->table_name . " 
                      (user_id, order_number, total_amount, currency, status, 
                       shipping_address, billing_address, payment_method, payment_status, notes,
                       voucher_id, voucher_discount_amount, voucher_discount_type, voucher_discount_value,
                       shipping_method_id) 
                      VALUES (:user_id, :order_number, :total_amount, :currency, :status, 
                              :shipping_address, :billing_address, :payment_method, :payment_status, :notes,
                              :voucher_id, :voucher_discount_amount, :voucher_discount_type, :voucher_discount_value,
                              :shipping_method_id)";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindValue(':user_id', $order_data['user_id'] ?: null);
            $stmt->bindValue(':order_number', $order_number);
            $stmt->bindValue(':total_amount', $order_data['total']);
            $stmt->bindValue(':currency', 'RM');
            $stmt->bindValue(':status', $order_data['initial_status'] ?? 'pending');
            $stmt->bindValue(':shipping_address', $shipping_address);
            $stmt->bindValue(':billing_address', $shipping_address); // Same as shipping for now
            $stmt->bindValue(':payment_method', $order_data['payment_method']);
            $stmt->bindValue(':payment_status', 'pending');
            $stmt->bindValue(':notes', $order_data['notes'] ?: null);
            $stmt->bindValue(':voucher_id', $order_data['voucher_id'] ?? null);
            $stmt->bindValue(':voucher_discount_amount', $order_data['voucher_discount_amount'] ?? 0.00);
            $stmt->bindValue(':voucher_discount_type', $order_data['voucher_discount_type'] ?? null);
            $stmt->bindValue(':voucher_discount_value', $order_data['voucher_discount_value'] ?? null);
            $stmt->bindValue(':shipping_method_id', $order_data['shipping_method_id'] ?? null);
            
            if (!$stmt->execute()) {
                $error_info = $stmt->errorInfo();
                error_log("Order creation error: " . $error_info[2]);
                $this->conn->rollBack();
                return false;
            }
            
            $order_id = $this->conn->lastInsertId();
            
            // Insert order items
            foreach ($order_data['items'] as $item) {
                $item_query = "INSERT INTO " . $this->order_items_table . " 
                               (order_id, product_id, product_name, product_price, 
                                quantity, size, version, total_price) 
                               VALUES (:order_id, :product_id, :product_name, :product_price, 
                                       :quantity, :size, :version, :total_price)";
                
                $item_stmt = $this->conn->prepare($item_query);
                $item_stmt->bindValue(':order_id', $order_id);
                $item_stmt->bindValue(':product_id', $item['product_id']);
                $item_stmt->bindValue(':product_name', $item['name']);
                $item_stmt->bindValue(':product_price', $item['price']);
                $item_stmt->bindValue(':quantity', $item['quantity']);
                $item_stmt->bindValue(':size', $item['size'] ?: null);
                $item_stmt->bindValue(':version', $item['version'] ?: null);
                $item_stmt->bindValue(':total_price', $item['price'] * $item['quantity']);
                
                if (!$item_stmt->execute()) {
                    $error_info = $item_stmt->errorInfo();
                    error_log("Order item creation error: " . $error_info[2]);
                    $this->conn->rollBack();
                    return false;
                }
            }
            
            // Group items by product_id and sum quantities
            $stock_deductions = [];
            foreach ($order_data['items'] as $item) {
                $product_id = $item['product_id'];
                if (isset($stock_deductions[$product_id])) {
                    $stock_deductions[$product_id] += $item['quantity'];
                } else {
                    $stock_deductions[$product_id] = $item['quantity'];
                }
            }
            
            // Debug: Log stock deduction summary
            error_log("Stock deduction summary:");
            foreach ($stock_deductions as $product_id => $total_quantity) {
                error_log("  Product ID $product_id: Total quantity to deduct = $total_quantity");
            }
            
            // Deduct stock for each unique product
            foreach ($stock_deductions as $product_id => $total_quantity) {
                $stock_query = "UPDATE products SET stock_quantity = stock_quantity - :quantity WHERE id = :product_id";
                $stock_stmt = $this->conn->prepare($stock_query);
                $stock_stmt->bindValue(':quantity', $total_quantity, PDO::PARAM_INT);
                $stock_stmt->bindValue(':product_id', $product_id, PDO::PARAM_INT);
                
                if (!$stock_stmt->execute()) {
                    $error_info = $stock_stmt->errorInfo();
                    error_log("Stock deduction error for product " . $product_id . ": " . $error_info[2]);
                    $this->conn->rollBack();
                    return false;
                }
                
                // Log stock deduction
                error_log("Stock deducted: Product ID " . $product_id . ", Total Quantity: " . $total_quantity);
            }
            
            $this->conn->commit();
            error_log("Order created successfully: ID " . $order_id);
            return $order_id;
            
        } catch (Exception $e) {
            $this->conn->rollBack();
            error_log("Order creation exception: " . $e->getMessage());
            error_log("Order creation trace: " . $e->getTraceAsString());
            return false;
        }
    }

    /**
     * Get order by ID
     */
    public function getById($order_id) {
        $query = "SELECT * FROM " . $this->table_name . " WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $order_id, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetch();
    }

    /**
     * Get order items
     */
    public function getOrderItems($order_id) {
        $query = "SELECT * FROM " . $this->order_items_table . " WHERE order_id = :order_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':order_id', $order_id, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Get order items for stock deduction
     */
    public function getOrderItemsForStock($order_id) {
        $query = "SELECT product_id, quantity, size, version 
                  FROM " . $this->order_items_table . " 
                  WHERE order_id = :order_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':order_id', $order_id, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Get orders by user
     */
    public function getByUser($user_id, $session_id = null) {
        $query = "SELECT * FROM " . $this->table_name . " WHERE 1=1";
        $params = [];
        
        if ($user_id) {
            $query .= " AND user_id = :user_id";
            $params['user_id'] = $user_id;
        } elseif ($session_id) {
            $query .= " AND session_id = :session_id";
            $params['session_id'] = $session_id;
        }
        
        $query .= " ORDER BY created_at DESC";
        
        $stmt = $this->conn->prepare($query);
        foreach ($params as $key => $value) {
            $stmt->bindValue(':' . $key, $value);
        }
        
        $stmt->execute();
        return $stmt->fetchAll();
    }

    /**
     * Update order status
     */
    public function updateStatus($order_id, $status, $tracking_number = null) {
        try {
            // Check if tracking_number column exists
            $columns_query = "SHOW COLUMNS FROM " . $this->table_name . " LIKE 'tracking_number'";
            $stmt = $this->conn->prepare($columns_query);
            $stmt->execute();
            $has_tracking_column = $stmt->rowCount() > 0;
            
            if ($has_tracking_column && $tracking_number !== null) {
                $query = "UPDATE " . $this->table_name . " SET status = :status, tracking_number = :tracking_number, updated_at = NOW() WHERE id = :id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindValue(':status', $status);
                $stmt->bindValue(':tracking_number', $tracking_number);
                $stmt->bindValue(':id', $order_id, PDO::PARAM_INT);
            } else {
                $query = "UPDATE " . $this->table_name . " SET status = :status, updated_at = NOW() WHERE id = :id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindValue(':status', $status);
                $stmt->bindValue(':id', $order_id, PDO::PARAM_INT);
            }
            
            return $stmt->execute();
        } catch (Exception $e) {
            error_log("Error updating order status: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Update order with payment information
     */
    public function updatePaymentInfo($order_id, $payment_data) {
        try {
            // Check which columns exist in the database
            $columns_query = "SHOW COLUMNS FROM " . $this->table_name;
            $stmt = $this->conn->prepare($columns_query);
            $stmt->execute();
            $columns = $stmt->fetchAll(PDO::FETCH_COLUMN);
            
            // Build dynamic query based on available data and existing columns
            $fields = [];
            $values = [];
            
            if (isset($payment_data['status']) && in_array('payment_status', $columns)) {
                $fields[] = "payment_status = :payment_status";
                $values[':payment_status'] = $payment_data['status'];
            }
            
            if (isset($payment_data['reference']) && in_array('payment_reference', $columns)) {
                $fields[] = "payment_reference = :payment_reference";
                $values[':payment_reference'] = $payment_data['reference'];
            }
            
            if (isset($payment_data['amount']) && in_array('payment_amount', $columns)) {
                $fields[] = "payment_amount = :payment_amount";
                $values[':payment_amount'] = $payment_data['amount'];
            }
            
            if (isset($payment_data['date']) && in_array('payment_date', $columns)) {
                $fields[] = "payment_date = :payment_date";
                $values[':payment_date'] = $payment_data['date'];
            }
            
            if (isset($payment_data['bill_code']) && in_array('bill_code', $columns)) {
                $fields[] = "bill_code = :bill_code";
                $values[':bill_code'] = $payment_data['bill_code'];
            }
            
            if (empty($fields)) {
                error_log("Order updatePaymentInfo: No valid fields to update for order $order_id");
                return false; // No fields to update
            }
            
            $fields[] = "updated_at = NOW()";
            $values[':id'] = $order_id;
            
            $query = "UPDATE " . $this->table_name . " SET " . implode(', ', $fields) . " WHERE id = :id";
            
            $stmt = $this->conn->prepare($query);
            
            foreach ($values as $key => $value) {
                $stmt->bindValue($key, $value);
            }
            
            $result = $stmt->execute();
            error_log("Order updatePaymentInfo: Updated order $order_id with fields: " . implode(', ', array_keys($payment_data)));
            return $result;
            
        } catch (Exception $e) {
            error_log("Order updatePaymentInfo error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get orders by payment status
     */
    public function getByPaymentStatus($user_id, $status) {
        $query = "SELECT * FROM " . $this->table_name . " WHERE user_id = :user_id AND status = :status ORDER BY created_at DESC";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $user_id);
        $stmt->bindValue(':status', $status);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }

    /**
     * Get user orders (alias for getByUser for consistency)
     */
    public function getUserOrders($user_id) {
        return $this->getByUser($user_id);
    }

    /**
     * Get order statistics
     */
    public function getStats($user_id = null, $session_id = null) {
        $query = "SELECT 
                    COUNT(*) as total_orders,
                    SUM(total) as total_spent,
                    AVG(total) as average_order
                  FROM " . $this->table_name . " WHERE 1=1";
        
        $params = [];
        
        if ($user_id) {
            $query .= " AND user_id = :user_id";
            $params['user_id'] = $user_id;
        } elseif ($session_id) {
            $query .= " AND session_id = :session_id";
            $params['session_id'] = $session_id;
        }
        
        $stmt = $this->conn->prepare($query);
        foreach ($params as $key => $value) {
            $stmt->bindValue(':' . $key, $value);
        }
        
        $stmt->execute();
        return $stmt->fetch();
    }

    /**
     * Get order items with product details for point calculation
     */
    public function getOrderItemsForPoints($order_id) {
        $query = "SELECT oi.product_id, oi.quantity, oi.size, oi.version, oi.product_price,
                         p.name as product_name
                  FROM " . $this->order_items_table . " oi
                  JOIN products p ON oi.product_id = p.id
                  WHERE oi.order_id = :order_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':order_id', $order_id, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

}
?>