<?php
require_once __DIR__ . '/../config/database.php';

class User {
    private $conn;
    private $table_name = "users";

    public function __construct() {
        $this->conn = getDB();
    }

    /**
     * Create new user
     */
    public function create($data) {
        $query = "INSERT INTO " . $this->table_name . " 
                  (email, password, first_name, last_name, phone, address, city, state, zip_code, country) 
                  VALUES (:email, :password, :first_name, :last_name, :phone, :address, :city, :state, :zip_code, :country)";
        
        $stmt = $this->conn->prepare($query);
        
        $stmt->bindValue(':email', $data['email']);
        $stmt->bindValue(':password', password_hash($data['password'], PASSWORD_DEFAULT));
        $stmt->bindValue(':first_name', $data['first_name']);
        $stmt->bindValue(':last_name', $data['last_name']);
        $stmt->bindValue(':phone', $data['phone'] ?? null);
        $stmt->bindValue(':address', $data['address'] ?? null);
        $stmt->bindValue(':city', $data['city'] ?? null);
        $stmt->bindValue(':state', $data['state'] ?? null);
        $stmt->bindValue(':zip_code', $data['zip_code'] ?? null);
        $stmt->bindValue(':country', $data['country'] ?? 'Malaysia');
        
        return $stmt->execute();
    }

    /**
     * Get user by email
     */
    public function getByEmail($email) {
        $query = "SELECT * FROM " . $this->table_name . " WHERE email = :email";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':email', $email);
        $stmt->execute();
        
        return $stmt->fetch();
    }

    /**
     * Get user by ID
     */
    public function getById($id) {
        $query = "SELECT * FROM " . $this->table_name . " WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetch();
    }

    /**
     * Verify user password
     */
    public function verifyPassword($email, $password) {
        $user = $this->getByEmail($email);
        
        if ($user && password_verify($password, $user['password'])) {
            return $user;
        }
        
        return false;
    }

    /**
     * Update user profile
     */
    public function updateProfile($id, $data) {
        $query = "UPDATE " . $this->table_name . " 
                  SET first_name = :first_name, last_name = :last_name, 
                      phone = :phone, address = :address, city = :city, 
                      state = :state, zip_code = :zip_code, country = :country 
                  WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        
        $stmt->bindValue(':first_name', $data['first_name']);
        $stmt->bindValue(':last_name', $data['last_name']);
        $stmt->bindValue(':phone', $data['phone'] ?? null);
        $stmt->bindValue(':address', $data['address'] ?? null);
        $stmt->bindValue(':city', $data['city'] ?? null);
        $stmt->bindValue(':state', $data['state'] ?? null);
        $stmt->bindValue(':zip_code', $data['zip_code'] ?? null);
        $stmt->bindValue(':country', $data['country'] ?? 'Malaysia');
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        
        return $stmt->execute();
    }

    /**
     * Change password
     */
    public function changePassword($id, $new_password) {
        $query = "UPDATE " . $this->table_name . " 
                  SET password = :password 
                  WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':password', password_hash($new_password, PASSWORD_DEFAULT));
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        
        return $stmt->execute();
    }

    /**
     * Update user avatar path
     */
    public function updateAvatar($id, $avatarPath) {
        $query = "UPDATE " . $this->table_name . " 
                  SET avatar = :avatar, updated_at = NOW() 
                  WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':avatar', $avatarPath);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        
        return $stmt->execute();
    }

    /**
     * Check if email exists
     */
    public function emailExists($email) {
        $query = "SELECT id FROM " . $this->table_name . " WHERE email = :email";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':email', $email);
        $stmt->execute();
        
        return $stmt->fetch() ? true : false;
    }

    /**
     * Create guest user
     */
    public function createGuest($session_id) {
        $query = "INSERT INTO " . $this->table_name . " 
                  (email, password, first_name, last_name, is_guest) 
                  VALUES (:email, :password, :first_name, :last_name, 1)";
        
        $stmt = $this->conn->prepare($query);
        
        $stmt->bindValue(':email', 'guest_' . $session_id . '@temp.com');
        $stmt->bindValue(':password', '');
        $stmt->bindValue(':first_name', 'Guest');
        $stmt->bindValue(':last_name', 'User');
        
        if ($stmt->execute()) {
            return $this->conn->lastInsertId();
        }
        
        return false;
    }

    /**
     * Generate a secure remember token
     */
    public function generateRememberToken() {
        return bin2hex(random_bytes(32));
    }

    /**
     * Create remember token for user
     */
    public function createRememberToken($user_id, $remember_me = false) {
        if (!$remember_me) {
            return null;
        }

        $token = $this->generateRememberToken();
        $expires_at = date('Y-m-d H:i:s', strtotime('+30 days')); // 30 days from now

        // Insert into remember_tokens table
        $query = "INSERT INTO remember_tokens (user_id, token, expires_at) VALUES (:user_id, :token, :expires_at)";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
        $stmt->bindValue(':token', $token);
        $stmt->bindValue(':expires_at', $expires_at);

        if ($stmt->execute()) {
            // Also update the remember_token field in users table for quick lookup
            $update_query = "UPDATE " . $this->table_name . " SET remember_token = :token WHERE id = :user_id";
            $update_stmt = $this->conn->prepare($update_query);
            $update_stmt->bindValue(':token', $token);
            $update_stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
            $update_stmt->execute();

            return $token;
        }

        return false;
    }

    /**
     * Verify remember token and get user
     */
    public function verifyRememberToken($token) {
        // First check if token exists in remember_tokens table and is not expired
        $query = "SELECT rt.user_id, rt.expires_at, u.* 
                  FROM remember_tokens rt 
                  JOIN " . $this->table_name . " u ON rt.user_id = u.id 
                  WHERE rt.token = :token AND rt.expires_at > NOW()";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':token', $token);
        $stmt->execute();
        
        $result = $stmt->fetch();
        
        if ($result) {
            // Token is valid, return user data
            return $result;
        }
        
        return false;
    }

    /**
     * Clear remember token
     */
    public function clearRememberToken($token) {
        // Get user_id from token
        $query = "SELECT user_id FROM remember_tokens WHERE token = :token";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':token', $token);
        $stmt->execute();
        $result = $stmt->fetch();

        if ($result) {
            $user_id = $result['user_id'];
            
            // Delete from remember_tokens table
            $delete_query = "DELETE FROM remember_tokens WHERE token = :token";
            $delete_stmt = $this->conn->prepare($delete_query);
            $delete_stmt->bindValue(':token', $token);
            $delete_stmt->execute();

            // Clear remember_token field in users table
            $update_query = "UPDATE " . $this->table_name . " SET remember_token = NULL WHERE id = :user_id";
            $update_stmt = $this->conn->prepare($update_query);
            $update_stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
            $update_stmt->execute();

            return true;
        }

        return false;
    }

    /**
     * Clear all remember tokens for a user
     */
    public function clearAllRememberTokens($user_id) {
        // Delete from remember_tokens table
        $delete_query = "DELETE FROM remember_tokens WHERE user_id = :user_id";
        $delete_stmt = $this->conn->prepare($delete_query);
        $delete_stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
        $delete_stmt->execute();

        // Clear remember_token field in users table
        $update_query = "UPDATE " . $this->table_name . " SET remember_token = NULL WHERE id = :user_id";
        $update_stmt = $this->conn->prepare($update_query);
        $update_stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
        $update_stmt->execute();

        return true;
    }

    /**
     * Clean up expired remember tokens
     */
    public function cleanupExpiredTokens() {
        $query = "DELETE FROM remember_tokens WHERE expires_at < NOW()";
        $stmt = $this->conn->prepare($query);
        return $stmt->execute();
    }

    /**
     * Get user's shipping addresses
     */
    public function getShippingAddresses($user_id) {
        $query = "SELECT * FROM shipping_addresses WHERE user_id = :user_id ORDER BY is_default DESC, created_at DESC";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }

    /**
     * Add shipping address
     */
    public function addShippingAddress($address_data) {
        try {
            $this->conn->beginTransaction();
            
            // If this is set as default, unset other defaults
            if ($address_data['is_default']) {
                $query = "UPDATE shipping_addresses SET is_default = 0 WHERE user_id = :user_id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindValue(':user_id', $address_data['user_id'], PDO::PARAM_INT);
                $stmt->execute();
            }
            
            $query = "INSERT INTO shipping_addresses 
                      (user_id, address_name, first_name, last_name, phone, address, city, state, zip_code, country, is_default) 
                      VALUES (:user_id, :address_name, :first_name, :last_name, :phone, :address, :city, :state, :zip_code, :country, :is_default)";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindValue(':user_id', $address_data['user_id'], PDO::PARAM_INT);
            $stmt->bindValue(':address_name', $address_data['address_name']);
            $stmt->bindValue(':first_name', $address_data['first_name']);
            $stmt->bindValue(':last_name', $address_data['last_name']);
            $stmt->bindValue(':phone', $address_data['phone']);
            $stmt->bindValue(':address', $address_data['address']);
            $stmt->bindValue(':city', $address_data['city']);
            $stmt->bindValue(':state', $address_data['state']);
            $stmt->bindValue(':zip_code', $address_data['zip_code']);
            $stmt->bindValue(':country', $address_data['country']);
            $stmt->bindValue(':is_default', $address_data['is_default'], PDO::PARAM_INT);
            
            $result = $stmt->execute();
            $this->conn->commit();
            
            return $result;
        } catch (Exception $e) {
            $this->conn->rollBack();
            error_log("Add shipping address error: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Update shipping address
     */
    public function updateShippingAddress($address_id, $address_data) {
        try {
            $this->conn->beginTransaction();
            
            // If this is set as default, unset other defaults
            if ($address_data['is_default']) {
                $query = "UPDATE shipping_addresses SET is_default = 0 WHERE user_id = (SELECT user_id FROM shipping_addresses WHERE id = :address_id)";
                $stmt = $this->conn->prepare($query);
                $stmt->bindValue(':address_id', $address_id, PDO::PARAM_INT);
                $stmt->execute();
            }
            
            $query = "UPDATE shipping_addresses 
                      SET address_name = :address_name, first_name = :first_name, last_name = :last_name, 
                          phone = :phone, address = :address, city = :city, state = :state, 
                          zip_code = :zip_code, country = :country, is_default = :is_default, updated_at = NOW()
                      WHERE id = :address_id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindValue(':address_name', $address_data['address_name']);
            $stmt->bindValue(':first_name', $address_data['first_name']);
            $stmt->bindValue(':last_name', $address_data['last_name']);
            $stmt->bindValue(':phone', $address_data['phone']);
            $stmt->bindValue(':address', $address_data['address']);
            $stmt->bindValue(':city', $address_data['city']);
            $stmt->bindValue(':state', $address_data['state']);
            $stmt->bindValue(':zip_code', $address_data['zip_code']);
            $stmt->bindValue(':country', $address_data['country']);
            $stmt->bindValue(':is_default', $address_data['is_default'], PDO::PARAM_INT);
            $stmt->bindValue(':address_id', $address_id, PDO::PARAM_INT);
            
            $result = $stmt->execute();
            $this->conn->commit();
            
            return $result;
        } catch (Exception $e) {
            $this->conn->rollBack();
            error_log("Update shipping address error: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Delete shipping address
     */
    public function deleteShippingAddress($address_id, $user_id) {
        $query = "DELETE FROM shipping_addresses WHERE id = :address_id AND user_id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':address_id', $address_id, PDO::PARAM_INT);
        $stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
        
        return $stmt->execute();
    }

    /**
     * Set default shipping address
     */
    public function setDefaultShippingAddress($address_id, $user_id) {
        try {
            $this->conn->beginTransaction();
            
            // Unset all defaults for this user
            $query = "UPDATE shipping_addresses SET is_default = 0 WHERE user_id = :user_id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
            $stmt->execute();
            
            // Set the specified address as default
            $query = "UPDATE shipping_addresses SET is_default = 1 WHERE id = :address_id AND user_id = :user_id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindValue(':address_id', $address_id, PDO::PARAM_INT);
            $stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
            
            $result = $stmt->execute();
            $this->conn->commit();
            
            return $result;
        } catch (Exception $e) {
            $this->conn->rollBack();
            error_log("Set default shipping address error: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Create or update user from Google OAuth
     */
    public function createOrUpdateFromGoogle($google_user) {
        try {
            // Check if user already exists
            $existing_user = $this->getByEmail($google_user['email']);
            
            if ($existing_user) {
                // Update existing user with Google info if not already set
                $query = "UPDATE " . $this->table_name . " 
                          SET google_id = :google_id, 
                              avatar = COALESCE(avatar, :avatar),
                              updated_at = NOW()
                          WHERE id = :id";
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindValue(':google_id', $google_user['id']);
                $stmt->bindValue(':avatar', $google_user['picture'] ?? null);
                $stmt->bindValue(':id', $existing_user['id'], PDO::PARAM_INT);
                
                if ($stmt->execute()) {
                    return $this->getById($existing_user['id']);
                }
                return false;
            } else {
                // Create new user from Google data
                $names = explode(' ', $google_user['name'], 2);
                $first_name = $names[0];
                $last_name = isset($names[1]) ? $names[1] : '';
                
                $query = "INSERT INTO " . $this->table_name . " 
                          (email, password, first_name, last_name, google_id, avatar, email_verified) 
                          VALUES (:email, :password, :first_name, :last_name, :google_id, :avatar, 1)";
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindValue(':email', $google_user['email']);
                $stmt->bindValue(':password', ''); // No password for OAuth users
                $stmt->bindValue(':first_name', $first_name);
                $stmt->bindValue(':last_name', $last_name);
                $stmt->bindValue(':google_id', $google_user['id']);
                $stmt->bindValue(':avatar', $google_user['picture'] ?? null);
                
                if ($stmt->execute()) {
                    $user_id = $this->conn->lastInsertId();
                    return $this->getById($user_id);
                }
                return false;
            }
        } catch (Exception $e) {
            error_log("Google OAuth user creation error: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Get user by Google ID
     */
    public function getByGoogleId($google_id) {
        $query = "SELECT * FROM " . $this->table_name . " WHERE google_id = :google_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':google_id', $google_id);
        $stmt->execute();
        
        return $stmt->fetch();
    }

    /**
     * Delete user account and all associated data
     */
    public function deleteAccount($user_id) {
        try {
            $this->conn->beginTransaction();
            
            // Get user data before deletion for file cleanup
            $user = $this->getById($user_id);
            
            // Delete from core tables that definitely exist
            $core_tables = [
                'cart' => 'user_id',
                'wishlist' => 'user_id',
                'reviews' => 'user_id'
            ];
            
            foreach ($core_tables as $table => $column) {
                $query = "DELETE FROM {$table} WHERE {$column} = :user_id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
                $stmt->execute();
            }
            
            // Delete from orders and order_items (handle foreign key constraints)
            try {
                // First delete order_items for this user's orders
                $query = "DELETE oi FROM order_items oi 
                         INNER JOIN orders o ON oi.order_id = o.id 
                         WHERE o.user_id = :user_id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
                $stmt->execute();
                
                // Then delete orders
                $query = "DELETE FROM orders WHERE user_id = :user_id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
                $stmt->execute();
            } catch (Exception $e) {
                error_log("Error deleting orders: " . $e->getMessage());
            }
            
            // Delete from optional tables if they exist
            $optional_tables = [
                'remember_tokens' => 'user_id',
                'shipping_addresses' => 'user_id',
                'user_loyalty_points' => 'user_id',
                'points_transactions' => 'user_id',
                'digital_stamps' => 'user_id',
                'referral_codes' => 'user_id',
                'referral_usage' => 'user_id'
            ];
            
            foreach ($optional_tables as $table => $column) {
                try {
                    $query = "DELETE FROM {$table} WHERE {$column} = :user_id";
                    $stmt = $this->conn->prepare($query);
                    $stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
                    $stmt->execute();
                } catch (Exception $e) {
                    // Table might not exist, continue
                    error_log("Optional table {$table} deletion failed: " . $e->getMessage());
                }
            }
            
            // Delete user's uploaded avatar file if exists
            if ($user && !empty($user['avatar']) && file_exists($user['avatar'])) {
                try {
                    unlink($user['avatar']);
                } catch (Exception $e) {
                    error_log("Error deleting avatar file: " . $e->getMessage());
                }
            }
            
            // Finally delete the user
            $query = "DELETE FROM " . $this->table_name . " WHERE id = :user_id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
            $result = $stmt->execute();
            
            if ($result) {
                $this->conn->commit();
                return true;
            } else {
                $this->conn->rollBack();
                return false;
            }
            
        } catch (Exception $e) {
            $this->conn->rollBack();
            error_log("Delete account error: " . $e->getMessage());
            return false;
        }
    }
}
?>