<?php
session_start();
require_once 'models/Order.php';
require_once 'config/payment.php';

$order_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$payment_status = $_GET['payment_status'] ?? '';

if (!$order_id) {
    header('Location: index.php');
    exit;
}

$order = new Order();
$order_data = $order->getById($order_id);
$order_items = $order->getOrderItems($order_id);

if (!$order_data) {
    header('Location: index.php');
    exit;
}

// Fetch product details for each order item to get current prices
require_once 'models/Product.php';
$product = new Product();

foreach ($order_items as $index => $item) {
    $product_data = $product->getById($item['product_id']);
    if ($product_data) {
        // Use the product's current price
        $order_items[$index]['current_price'] = $product_data['price'];
        $order_items[$index]['current_total'] = $product_data['price'] * $item['quantity'];
        // Add product image to the item data
        $order_items[$index]['product_image'] = $product_data['image'] ?? 'images/placeholder.svg';
        // Ensure we have the product name
        if (empty($item['product_name'])) {
            $order_items[$index]['product_name'] = $product_data['name'];
        }
        
    }
}

// Get voucher information if voucher was used
$voucher_info = null;
if (!empty($order_data['voucher_id'])) {
    require_once 'admin/includes/Voucher.php';
    $voucherModel = new Voucher();
    $voucher_info = $voucherModel->getVoucherById($order_data['voucher_id']);
}

// Debug: Log voucher information
error_log("Order Success Debug - Order ID: " . $order_id);
error_log("Order Success Debug - Voucher ID: " . ($order_data['voucher_id'] ?? 'NULL'));
error_log("Order Success Debug - Voucher Discount Amount: " . ($order_data['voucher_discount_amount'] ?? 'NULL'));
error_log("Order Success Debug - Voucher Info: " . ($voucher_info ? 'Found' : 'Not found'));

// Auto-update database if payment is successful
if ($payment_status === '1') {
    // Payment successful - update database
    $order->updateStatus($order_id, 'processing');
    $order->updatePaymentInfo($order_id, [
        'status' => 'paid'
    ]);
    
    // Log the update
    error_log("Order Success Page: Auto-updated order #$order_id to processing/paid status");
    
    // Process stock deduction and points awarding
    try {
        // Check if already processed to avoid duplication
        require_once 'config/database.php';
        $conn = getDB();
        $stmt = $conn->prepare("SELECT * FROM points_transactions WHERE order_id = ? AND points > 0");
        $stmt->execute([$order_id]);
        $existing_points = $stmt->fetch();
        
        if (!$existing_points) {
            // Deduct stock for order items
            require_once 'models/Product.php';
            $product = new Product();
            
            $order_items = $order->getOrderItemsForStock($order_id);
            if (!empty($order_items)) {
                $stock_results = $product->deductOrderStock($order_items);
                
                $success_count = 0;
                $fail_count = 0;
                
                foreach ($stock_results as $result) {
                    if ($result['success']) {
                        $success_count++;
                    } else {
                        $fail_count++;
                        error_log("Stock deduction failed for product_id={$result['product_id']}, size={$result['size']}, version={$result['version']}, quantity={$result['quantity']}");
                    }
                }
                
                error_log("Stock deduction completed for order {$order_id}: {$success_count} successful, {$fail_count} failed");
            }
            
            // Award loyalty points for purchased products
            require_once 'models/Loyalty.php';
            $loyalty = new Loyalty();
            
            $order_items = $order->getOrderItemsForPoints($order_id);
            if (!empty($order_items)) {
                $total_points = 0;
                $points_awarded = [];
                
                foreach ($order_items as $item) {
                    // Calculate points based on price (RM1 = 1 point) instead of manual point_rewards
                    $item_price = $item['product_price'] ?? 0;
                    if ($item_price > 0) {
                        $points_per_item = $loyalty->calculatePointsFromPrice($item_price);
                        $total_points += $points_per_item * $item['quantity'];
                        $points_awarded[] = [
                            'product_name' => $item['product_name'],
                            'quantity' => $item['quantity'],
                            'points_per_item' => $points_per_item,
                            'total_points' => $points_per_item * $item['quantity']
                        ];
                    }
                }
                
                if ($total_points > 0) {
                    // Award points to user
                    $user_id = $order_data['user_id'];
                    $description = "Purchase reward for Order #{$order_id}";
                    
                    $points_result = $loyalty->addPoints($user_id, $total_points, 'earned', $description, $order_id, 'ORDER' . $order_id);
                    
                    if ($points_result) {
                        error_log("Loyalty points awarded for order {$order_id}: {$total_points} points to user {$user_id}");
                        
                        // Log detailed points breakdown
                        foreach ($points_awarded as $award) {
                            error_log("Points breakdown: {$award['product_name']} x{$award['quantity']} = {$award['total_points']} points ({$award['points_per_item']} per item)");
                        }
                    } else {
                        error_log("Failed to award loyalty points for order {$order_id}");
                    }
                } else {
                    error_log("No loyalty points to award for order {$order_id} (no products with point rewards)");
                }
            } else {
                error_log("No order items found for point calculation for order {$order_id}");
            }
        } else {
            error_log("Order {$order_id} already processed - skipping stock deduction and points awarding");
        }
    } catch (Exception $e) {
        error_log("Error processing order {$order_id}: " . $e->getMessage());
    }
    
    // Refresh order data to get updated status
    $order_data = $order->getById($order_id);
    
    // Refresh order items to get updated data
    $order_items = $order->getOrderItems($order_id);
    
    // Re-fetch product details for each order item to get current prices
    foreach ($order_items as $index => $item) {
        $product_data = $product->getById($item['product_id']);
        if ($product_data) {
            // Use the product's current price
            $order_items[$index]['current_price'] = $product_data['price'];
            $order_items[$index]['current_total'] = $product_data['price'] * $item['quantity'];
            // Add product image to the item data
            $order_items[$index]['product_image'] = $product_data['image'] ?? 'images/placeholder.svg';
            // Ensure we have the product name
            if (empty($item['product_name'])) {
                $order_items[$index]['product_name'] = $product_data['name'];
            }
            
        }
    }
}

// Handle payment status display
$payment_status_message = '';
$payment_status_class = '';
$display_payment_status = '';

// Determine display status based on payment status
$display_order_status = $order_data['status'] ?? 'awaiting_payment';

if ($payment_status) {
    switch ($payment_status) {
        case '1':
            $payment_status_message = 'Payment Successful!';
            $payment_status_class = 'success';
            $display_payment_status = 'paid';
            $display_order_status = 'processing'; // Payment successful = order processing
            break;
        case '2':
        case '4':
            $payment_status_message = 'Payment Pending';
            $payment_status_class = 'warning';
            $display_payment_status = 'pending';
            $display_order_status = 'awaiting_payment';
            break;
        case '3':
            $payment_status_message = 'Payment Failed';
            $payment_status_class = 'error';
            $display_payment_status = 'failed';
            $display_order_status = 'awaiting_payment';
            break;
    }
} else {
    // Fallback to order data payment status
    $order_payment_status = $order_data['payment_status'] ?? 'pending';
    switch ($order_payment_status) {
        case 'paid':
            $payment_status_message = 'Payment Successful!';
            $payment_status_class = 'success';
            $display_payment_status = 'paid';
            $display_order_status = 'processing'; // Payment successful = order processing
            break;
        case 'pending':
            $payment_status_message = 'Payment Pending';
            $payment_status_class = 'warning';
            $display_payment_status = 'pending';
            $display_order_status = 'awaiting_payment';
            break;
        case 'failed':
            $payment_status_message = 'Payment Failed';
            $payment_status_class = 'error';
            $display_payment_status = 'failed';
            $display_order_status = 'awaiting_payment';
            break;
        default:
            $payment_status_message = 'Payment Pending';
            $payment_status_class = 'warning';
            $display_payment_status = 'pending';
            $display_order_status = 'awaiting_payment';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Order Confirmation - 2 Days Parfume</title>
    <link rel="stylesheet" href="styles.css">
    <link rel="stylesheet" href="checkout.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="script.js" defer></script>
</head>
<body>
    <!-- Top Navigation -->
    <nav class="top-nav">
        <div class="nav-left">
            <i class="fas fa-arrow-left" onclick="window.location.href='index.php'"></i>
        </div>
        <div class="nav-center">
            <div class="logo">
                <img src="images/LOGO.png" alt="2 Days Parfume Logo" class="logo-img">
            </div>
        </div>
        <div class="nav-right">
            <div class="cart-icon">
                <i class="fas fa-shopping-cart"></i>
                <span class="cart-count" style="display: none;">0</span>
            </div>
        </div>
    </nav>

    <!-- Success Header -->
    <div class="checkout-header">
        <div class="success-icon">
            <i class="fas fa-check-circle"></i>
        </div>
        <h1>Order Confirmed!</h1>
        <p class="checkout-subtitle">Thank you for your purchase</p>
        
        <?php if ($payment_status_message): ?>
            <div class="payment-status-card payment-<?php echo $payment_status_class; ?>">
                <div class="payment-status-icon">
                    <i class="fas fa-<?php echo $payment_status_class === 'success' ? 'check-circle' : ($payment_status_class === 'warning' ? 'clock' : 'exclamation-circle'); ?>"></i>
                </div>
                <div class="payment-status-content">
                    <h3><?php echo $payment_status_message; ?></h3>
                    <?php if ($payment_status_class === 'success'): ?>
                        <p>Your payment has been processed successfully. You will receive a confirmation email shortly.</p>
                    <?php elseif ($payment_status_class === 'warning'): ?>
                        <p>Your payment is being processed. Please wait for confirmation or complete the payment if redirected.</p>
                    <?php else: ?>
                        <p>There was an issue processing your payment. Please try again or contact support.</p>
                    <?php endif; ?>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <main class="checkout-main">
        <div class="checkout-container">
            <!-- Order Details -->
            <div class="order-details-section">
                <div class="order-info">
                    <h2 class="section-title">
                        <i class="fas fa-receipt"></i>
                        Order Details
                    </h2>
                    
                    <div class="order-info-grid">
                        <div class="info-item">
                            <label>Order Number</label>
                            <span><?php echo $order_data['order_number']; ?></span>
                        </div>
                        <div class="info-item">
                            <label>Order Date</label>
                            <span><?php echo date('M d, Y', strtotime($order_data['created_at'])); ?></span>
                        </div>
                        <div class="info-item">
                            <label>Status</label>
                            <span class="status-badge status-<?php echo $display_order_status; ?>">
                                <?php echo ucfirst(str_replace('_', ' ', $display_order_status)); ?>
                            </span>
                        </div>
                        <div class="info-item">
                            <label>Payment Method</label>
                            <span><?php echo getPaymentMethodName($order_data['payment_method']); ?></span>
                        </div>
                        <div class="info-item">
                            <label>Payment Status</label>
                            <span class="status-badge status-<?php echo $display_payment_status; ?>">
                                <?php echo ucfirst($display_payment_status); ?>
                            </span>
                        </div>
                        <?php if (!empty($order_data['shipping_method'])): ?>
                        <div class="info-item">
                            <label>Shipping Method</label>
                            <span><?php echo htmlspecialchars($order_data['shipping_method']); ?></span>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Shipping Information -->
                <div class="shipping-info">
                    <h2 class="section-title">
                        <i class="fas fa-truck"></i>
                        Shipping Information
                    </h2>
                    
                    <div class="shipping-details">
                        <?php 
                        $shipping_address = json_decode($order_data['shipping_address'], true);
                        if ($shipping_address):
                        ?>
                            <p><strong><?php echo htmlspecialchars($shipping_address['first_name'] . ' ' . $shipping_address['last_name']); ?></strong></p>
                            <p><?php echo htmlspecialchars($shipping_address['address']); ?></p>
                            <p><?php echo htmlspecialchars($shipping_address['city'] . ', ' . $shipping_address['state'] . ' ' . $shipping_address['zip_code']); ?></p>
                            <p><?php echo htmlspecialchars($shipping_address['country']); ?></p>
                            <p><strong>Phone:</strong> <?php echo htmlspecialchars($shipping_address['phone']); ?></p>
                            <p><strong>Email:</strong> <?php echo htmlspecialchars($shipping_address['email']); ?></p>
                        <?php else: ?>
                            <p>Shipping address not available</p>
                        <?php endif; ?>
                        
                        <?php if (!empty($order_data['notes'])): ?>
                        <div class="order-notes" style="margin-top: 1rem; padding: 1rem; background: #f8f9fa; border-radius: 8px; border-left: 4px solid #007bff;">
                            <h4 style="margin: 0 0 0.5rem 0; color: #495057; font-size: 0.9rem;">Order Notes:</h4>
                            <p style="margin: 0; color: #6c757d; font-style: italic;"><?php echo htmlspecialchars($order_data['notes']); ?></p>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Order Items -->
                <div class="order-items-section">
                    <h2 class="section-title">
                        <i class="fas fa-shopping-bag"></i>
                        Order Items
                    </h2>
                    
                    
                    <div class="order-items">
                        <?php foreach ($order_items as $item): ?>
                        <div class="order-item">
                            <div class="item-image">
                                <?php 
                                // Get the actual product image from the product data
                                $product_image = $item['product_image'] ?? 'images/placeholder.svg';
                                
                                ?>
                                <img src="<?php echo htmlspecialchars($product_image); ?>" 
                                     alt="<?php echo htmlspecialchars($item['product_name'] ?? 'Product'); ?>" 
                                     onerror="this.src='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMTAwIiBoZWlnaHQ9IjEwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMTAwIiBoZWlnaHQ9IjEwMCIgZmlsbD0iI2Y0ZjRmNCIvPjx0ZXh0IHg9IjUwIiB5PSI1MCIgZm9udC1mYW1pbHk9IkFyaWFsIiBmb250LXNpemU9IjE0IiBmaWxsPSIjOTk5IiB0ZXh0LWFuY2hvcj0ibWlkZGxlIj5JbWFnZTwvdGV4dD48L3N2Zz4='">
                            </div>
                            <div class="item-details">
                                <h3 class="item-name"><?php echo htmlspecialchars($item['product_name'] ?? 'Product'); ?></h3>
                                <?php if ($item['size'] || $item['version']): ?>
                                    <p class="item-size">
                                        <?php if (!empty($item['version'])): ?>
                                            Version: <?php echo htmlspecialchars($item['version']); ?> | 
                                        <?php endif; ?>
                                        <?php if (!empty($item['size'])): ?>
                                            Size: <?php echo htmlspecialchars($item['size']); ?>
                                        <?php endif; ?>
                                    </p>
                                <?php endif; ?>
                                <p class="item-quantity">Qty: <?php echo $item['quantity']; ?></p>
                                <div class="item-price">
                                    RM <?php echo number_format($item['product_price'] ?? $item['subtotal'] ?? 0, 2); ?>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>


                    <!-- Order Totals -->
                    <div class="order-totals">
                        <?php if ($voucher_info && $order_data['voucher_discount_amount'] > 0): ?>
                        <div class="total-row voucher-discount">
                            <div class="voucher-info">
                                <span>Voucher Applied</span>
                                <div class="voucher-details">
                                    <small class="voucher-name"><?php echo htmlspecialchars($voucher_info['voucher_name']); ?></small>
                                    <span class="voucher-type">
                                        <?php 
                                        if ($order_data['voucher_discount_type'] === 'percentage') {
                                            echo $order_data['voucher_discount_value'] . '% OFF';
                                        } elseif ($order_data['voucher_discount_type'] === 'fixed_amount') {
                                            echo 'RM ' . $order_data['voucher_discount_value'] . ' OFF';
                                        } else {
                                            echo ucfirst(str_replace('_', ' ', $order_data['voucher_discount_type']));
                                        }
                                        ?>
                                    </span>
                                </div>
                            </div>
                            <span class="discount-amount">
                                -RM <?php echo number_format($order_data['voucher_discount_amount'], 2); ?>
                            </span>
                        </div>
                        <?php endif; ?>
                        
                        <div class="total-row">
                            <span>Total</span>
                            <span>RM <?php echo number_format($order_data['total_amount'], 2); ?></span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Next Steps -->
            <div class="next-steps-section">
                <div class="next-steps">
                    <h2 class="section-title">
                        <i class="fas fa-info-circle"></i>
                        What's Next?
                    </h2>
                    
                    <div class="steps-list">
                        <div class="step">
                            <div class="step-icon">
                                <i class="fas fa-clock"></i>
                            </div>
                            <div class="step-content">
                                <h3>Order Processing</h3>
                                <p>We'll process your order within 1-2 business days.</p>
                            </div>
                        </div>
                        
                        <div class="step">
                            <div class="step-icon">
                                <i class="fas fa-shipping-fast"></i>
                            </div>
                            <div class="step-content">
                                <h3>Shipping</h3>
                                <p>Your order will be shipped within 3-5 business days.</p>
                            </div>
                        </div>
                        
                        <div class="step">
                            <div class="step-icon">
                                <i class="fas fa-home"></i>
                            </div>
                            <div class="step-content">
                                <h3>Delivery</h3>
                                <p>You'll receive your order within 5-7 business days.</p>
                            </div>
                        </div>
                    </div>

                    <!-- Action Buttons -->
                    <div class="action-buttons">
                        <button class="btn btn-primary" onclick="window.location.href='index.php'">
                            <i class="fas fa-home"></i>
                            Continue Shopping
                        </button>
                        <button class="btn btn-secondary" onclick="window.location.href='profile.php'">
                            <i class="fas fa-user"></i>
                            View Profile
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <!-- Bottom Navigation -->
    <nav class="bottom-nav">
        <div class="nav-item" onclick="window.location.href='index.php'">
            <i class="fas fa-home"></i>
        </div>
        <div class="nav-item" onclick="window.location.href='products.php'">
            <i class="fas fa-box"></i>
        </div>
        <div class="nav-item" onclick="window.location.href='cart.php'">
            <i class="fas fa-shopping-cart"></i>
        </div>
        <div class="nav-item" onclick="checkRegisteredUserAndRedirect('royalty-rewards.php')">
            <i class="fas fa-crown"></i>
        </div>
    </nav>

    <style>
        .success-icon {
            font-size: 4rem;
            color: #28a745;
            margin-bottom: 1rem;
        }
        
        /* Professional Payment Status Card */
        .payment-status-card {
            background: #fff;
            border-radius: 12px;
            padding: 1.5rem;
            margin: 1.5rem 0;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            border-left: 4px solid;
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .payment-status-card.payment-success {
            border-left-color: #28a745;
            background: linear-gradient(135deg, #f8fff9 0%, #e8f5e8 100%);
        }
        
        .payment-status-card.payment-warning {
            border-left-color: #ffc107;
            background: linear-gradient(135deg, #fffdf7 0%, #fef7e0 100%);
        }
        
        .payment-status-card.payment-error {
            border-left-color: #dc3545;
            background: linear-gradient(135deg, #fff8f8 0%, #ffeaea 100%);
        }
        
        .payment-status-icon {
            font-size: 2.5rem;
            flex-shrink: 0;
        }
        
        .payment-status-card.payment-success .payment-status-icon {
            color: #28a745;
        }
        
        .payment-status-card.payment-warning .payment-status-icon {
            color: #ffc107;
        }
        
        .payment-status-card.payment-error .payment-status-icon {
            color: #dc3545;
        }
        
        .payment-status-content h3 {
            margin: 0 0 0.5rem 0;
            font-size: 1.25rem;
            font-weight: 600;
            color: #2c3e50;
        }
        
        .payment-status-content p {
            margin: 0;
            color: #6c757d;
            font-size: 0.95rem;
            line-height: 1.4;
        }
        
        /* Improved spacing and margins */
        .checkout-main {
            padding: 1rem;
            max-width: 100%;
            margin: 0 auto;
        }
        
        .checkout-container {
            background: #fff;
            border-radius: 12px;
            padding: 1.5rem;
            margin: 1rem 0;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        }
        
        .order-details, .shipping-info, .order-items, .order-totals {
            margin-bottom: 2rem;
            padding: 1.5rem;
            background: #f8f9fa;
            border-radius: 8px;
            border: 1px solid #e9ecef;
        }
        
        .order-details:last-child, .shipping-info:last-child, .order-items:last-child, .order-totals:last-child {
            margin-bottom: 0;
        }
        
        .section-title {
            margin-bottom: 1rem;
            color: #2c3e50;
            font-size: 1.1rem;
            font-weight: 600;
        }
        
        .info-item {
            margin-bottom: 1rem;
            padding-bottom: 0.5rem;
            border-bottom: 1px solid #e9ecef;
        }
        
        .info-item:last-child {
            margin-bottom: 0;
            border-bottom: none;
        }
        
        .info-item label {
            display: block;
            font-weight: 600;
            color: #495057;
            margin-bottom: 0.25rem;
            font-size: 0.9rem;
        }
        
        .info-item span {
            color: #2c3e50;
            font-size: 0.95rem;
        }
        
        .order-info-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1rem;
            margin-bottom: 2rem;
        }
        
        .info-item {
            display: flex;
            flex-direction: column;
            gap: 0.25rem;
        }
        
        .info-item label {
            font-size: 0.8rem;
            color: #666;
            font-weight: 500;
        }
        
        .info-item span {
            font-weight: 600;
            color: #333;
        }
        
        .status-badge {
            padding: 0.5rem 1rem;
            border-radius: 25px;
            font-size: 0.85rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }
        
        .status-badge::before {
            content: '';
            width: 8px;
            height: 8px;
            border-radius: 50%;
            background: currentColor;
        }
        
        .status-pending {
            background: linear-gradient(135deg, #fff3cd 0%, #ffeaa7 100%);
            color: #856404;
            border: 1px solid #ffeaa7;
        }
        
        .status-processing {
            background: linear-gradient(135deg, #d1ecf1 0%, #bee5eb 100%);
            color: #0c5460;
            border: 1px solid #bee5eb;
        }
        
        .status-shipped {
            background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .status-delivered {
            background: linear-gradient(135deg, #d1ecf1 0%, #bee5eb 100%);
            color: #0c5460;
            border: 1px solid #bee5eb;
        }
        
        .status-paid {
            background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .status-failed {
            background: linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%);
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .status-processing {
            background: linear-gradient(135deg, #cce5ff 0%, #b3d9ff 100%);
            color: #004085;
            border: 1px solid #b3d9ff;
        }
        
        .payment-status-message {
            margin-top: 1rem;
            padding: 1rem;
            border-radius: 8px;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-weight: 600;
        }
        
        .payment-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .payment-warning {
            background: #fff3cd;
            color: #856404;
            border: 1px solid #ffeaa7;
        }
        
        .payment-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .shipping-details {
            background: #f8f9fa;
            padding: 1rem;
            border-radius: 8px;
            line-height: 1.6;
        }
        
        .steps-list {
            margin-bottom: 2rem;
        }
        
        .step {
            display: flex;
            align-items: flex-start;
            gap: 1rem;
            margin-bottom: 1.5rem;
        }
        
        .step-icon {
            width: 40px;
            height: 40px;
            background: #FFD700;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #8B4513;
            flex-shrink: 0;
        }
        
        .step-content h3 {
            margin: 0 0 0.25rem 0;
            font-size: 1rem;
            color: #333;
        }
        
        .step-content p {
            margin: 0;
            color: #666;
            font-size: 0.9rem;
        }
        
        .action-buttons {
            display: flex;
            flex-direction: column;
            gap: 1rem;
        }
        
        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #8B4513 0%, #A0522D 100%);
            color: white;
        }
        
        .btn-primary:hover {
            background: linear-gradient(135deg, #A0522D 0%, #8B4513 100%);
            transform: translateY(-2px);
        }
        
        .btn-secondary {
            background: white;
            color: #8B4513;
            border: 2px solid #8B4513;
        }
        
        .btn-secondary:hover {
            background: #8B4513;
            color: white;
        }
        
        /* Voucher Discount Styles */
        .voucher-discount {
            background: linear-gradient(135deg, #f0fff4, #e6fffa);
            border: 2px solid #9ae6b4;
            border-radius: 8px;
            padding: 12px 16px;
            margin: 8px 0;
            box-shadow: 0 2px 8px rgba(46, 160, 67, 0.1);
        }
        
        .voucher-discount .voucher-info {
            display: flex;
            flex-direction: column;
            gap: 4px;
        }
        
        .voucher-discount .voucher-info span:first-child {
            font-size: 1rem;
            color: #1a4d2a;
            font-weight: 600;
        }
        
        .voucher-discount .voucher-details {
            display: flex;
            flex-direction: column;
            gap: 4px;
            margin-top: 4px;
        }
        
        .voucher-discount .voucher-name {
            font-size: 0.85rem;
            color: #2f855a;
            font-weight: 500;
        }
        
        .voucher-discount .voucher-type {
            font-size: 0.9rem;
            color: #1a4d2a;
            font-weight: 600;
            background: linear-gradient(135deg, #2f855a, #38a169);
            color: white;
            padding: 4px 10px;
            border-radius: 15px;
            display: inline-block;
            box-shadow: 0 2px 4px rgba(46, 160, 67, 0.3);
        }
        
        .voucher-discount .discount-amount {
            color: #1a4d2a;
            font-weight: 700;
            font-size: 1.2rem;
            text-shadow: 0 1px 2px rgba(0, 0, 0, 0.1);
        }
        
        /* Order Items Styling */
        .order-item {
            display: flex;
            align-items: center;
            gap: 1rem;
            padding: 1rem;
            background: white;
            border-radius: 8px;
            border: 1px solid #e9ecef;
            margin-bottom: 1rem;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
        }
        
        .order-item:last-child {
            margin-bottom: 0;
        }
        
        .item-image {
            width: 80px;
            height: 80px;
            border-radius: 8px;
            overflow: hidden;
            flex-shrink: 0;
            background: #f8f9fa;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .item-image img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .item-details {
            flex: 1;
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }
        
        .item-name {
            margin: 0;
            font-size: 1.1rem;
            font-weight: 600;
            color: #2c3e50;
            line-height: 1.3;
        }
        
        .item-size {
            margin: 0;
            font-size: 0.9rem;
            color: #6c757d;
            font-weight: 500;
        }
        
        .item-quantity {
            margin: 0;
            font-size: 0.9rem;
            color: #495057;
            font-weight: 500;
        }
        
        .item-price {
            font-size: 1.1rem;
            font-weight: 700;
            color: #28a745;
            margin-top: 0.5rem;
        }
        
        @media (min-width: 768px) {
            .order-info-grid {
                grid-template-columns: 1fr 1fr 1fr 1fr;
            }
            
            .action-buttons {
                flex-direction: row;
            }
        }
    </style>
</body>
</html>