<?php
require_once 'models/Product.php';
require_once 'models/Loyalty.php';
require_once 'config/database.php';

// Get search query and category from URL parameters
$searchQuery = isset($_GET['search']) ? $_GET['search'] : '';
$category = isset($_GET['category']) ? $_GET['category'] : 'all';

// Get products from database
$product = new Product();
$loyalty = new Loyalty();
$filters = [];

if ($searchQuery) {
    $filters['search'] = $searchQuery;
}

if ($category !== 'all') {
    $filters['category'] = $category;
}

$allProducts = $product->getAll($filters);
$filteredProducts = $allProducts;

// Get categories from database
$conn = getDB();

// Check if sort_order column exists, if not use id for ordering
$checkColumnQuery = "SHOW COLUMNS FROM categories LIKE 'sort_order'";
$checkStmt = $conn->query($checkColumnQuery);
$sortOrderExists = $checkStmt->rowCount() > 0;

if ($sortOrderExists) {
    $categoriesQuery = "SELECT * FROM categories WHERE is_active = 1 ORDER BY sort_order ASC, name ASC";
} else {
    $categoriesQuery = "SELECT * FROM categories WHERE is_active = 1 ORDER BY id ASC, name ASC";
}

$categoriesStmt = $conn->query($categoriesQuery);
$categories = $categoriesStmt->fetchAll();

// Debug: Check if categories are being fetched
if (empty($categories)) {
    // Try without is_active filter in case the column doesn't exist
    $fallbackQuery = "SELECT * FROM categories ORDER BY id ASC, name ASC";
    $fallbackStmt = $conn->query($fallbackQuery);
    $categories = $fallbackStmt->fetchAll();
}

// Group products by category
$productsByCategory = [];
foreach ($allProducts as $product) {
    $categorySlug = $product['category_slug'] ?? 'uncategorized';
    if (!isset($productsByCategory[$categorySlug])) {
        $productsByCategory[$categorySlug] = [
            'category' => $product['category_name'] ?? 'Uncategorized',
            'products' => []
        ];
    }
    $productsByCategory[$categorySlug]['products'][] = $product;
}

// Get best sellers and new arrivals
$bestSellers = array_filter($allProducts, function($product) {
    return $product['is_bestseller'] == 1;
});

$newArrivals = array_filter($allProducts, function($product) {
    return $product['is_new_arrival'] == 1;
});
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Products - 2 Days Parfume</title>
    <link rel="stylesheet" href="styles.css">
    <link rel="stylesheet" href="products.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <?php include_once 'includes/appilix-sdk.php'; ?>
    <?php include_once 'includes/appilix-user-identity.php'; ?>
    <script src="script.js" defer></script>
</head>
<body>
    <!-- Top Navigation -->
    <nav class="top-nav">
        <div class="nav-left">
            <i class="fas fa-arrow-left" onclick="goBack()"></i>
        </div>
        <div class="nav-center">
            <div class="logo">
                <img src="images/LOGO.png" alt="2 Days Parfume Logo" class="logo-img">
            </div>
        </div>
        <div class="nav-right">
            <div class="cart-icon" onclick="checkAuthAndRedirect('cart.php')">
                <i class="fas fa-shopping-cart"></i>
                <span class="cart-count">0</span>
            </div>
        </div>
    </nav>

    <!-- Search Bar -->
    <div class="search-section">
        <div class="search-container">
            <input type="text" id="search-input" placeholder="Search..." value="<?php echo htmlspecialchars($searchQuery); ?>">
            <?php if (!empty($searchQuery)): ?>
                <i class="fas fa-times clear-search-icon" onclick="clearSearch()" title="Clear search"></i>
            <?php endif; ?>
            <i class="fas fa-search search-icon"></i>
        </div>
        <?php if (!empty($searchQuery)): ?>
            <div class="search-results-info">
                <p>Search results for: "<strong><?php echo htmlspecialchars($searchQuery); ?></strong>"</p>
                <p><?php echo count($filteredProducts); ?> product(s) found</p>
            </div>
        <?php endif; ?>
    </div>


    <!-- Product Sections -->
    <main class="products-main">
        <!-- Best Sellers Section (if any best sellers exist) -->
        <?php if (!empty($bestSellers)): ?>
        <section class="product-section">
            <div class="section-header">
                <div class="section-title">
                    <h2>Best Sellers</h2>
                    <p class="section-subtitle">The Best Parfume Ever</p>
                </div>
                <button class="see-all-btn" onclick="window.location.href='product-catalog.php?category=bestseller'">see all ></button>
            </div>
            
            <div class="products-scroll-container">
                <div class="products-scroll">
                    <?php foreach (array_slice($bestSellers, 0, 4) as $product): ?>
                        <?php include 'includes/product-card.php'; ?>
                    <?php endforeach; ?>
                </div>
            </div>
        </section>
        <?php endif; ?>

        <!-- Just Arrived Section (if any new arrivals exist) -->
        <?php if (!empty($newArrivals)): ?>
        <section class="product-section">
            <div class="section-header">
                <div class="section-title">
                    <h2>Just Arrived</h2>
                    <p class="section-subtitle">Recently Arrived Parfums</p>
                </div>
                <button class="see-all-btn" onclick="window.location.href='product-catalog.php?category=new'">see all ></button>
            </div>
            
            <div class="products-scroll-container">
                <div class="products-scroll">
                    <?php foreach (array_slice($newArrivals, 0, 4) as $product): ?>
                        <?php include 'includes/product-card.php'; ?>
                    <?php endforeach; ?>
                </div>
            </div>
        </section>
        <?php endif; ?>

        <!-- Dynamic Category Sections -->
        <?php if (!empty($categories)): ?>
            <?php foreach ($categories as $category): ?>
                <?php 
                $categorySlug = $category['slug'];
                $categoryProducts = $productsByCategory[$categorySlug]['products'] ?? [];
                ?>
                <?php if (!empty($categoryProducts)): ?>
                <section class="product-section">
                    <div class="section-header">
                        <div class="section-title">
                            <h2><?php echo htmlspecialchars($category['name']); ?></h2>
                            <?php if (!empty($category['description'])): ?>
                                <p class="section-subtitle"><?php echo htmlspecialchars($category['description']); ?></p>
                            <?php endif; ?>
                        </div>
                        <button class="see-all-btn" onclick="window.location.href='product-catalog.php?category=<?php echo $categorySlug; ?>'">see all ></button>
                    </div>
                    
                    <div class="products-scroll-container">
                        <div class="products-scroll">
                            <?php foreach (array_slice($categoryProducts, 0, 4) as $product): ?>
                                <?php include 'includes/product-card.php'; ?>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </section>
                <?php endif; ?>
            <?php endforeach; ?>
        <?php else: ?>
            <!-- Fallback: Show all products if no categories are set up -->
            <section class="product-section">
                <div class="section-header">
                    <div class="section-title">
                        <h2>All Products</h2>
                        <p class="section-subtitle">Browse our complete collection</p>
                    </div>
                    <button class="see-all-btn" onclick="window.location.href='product-catalog.php'">see all ></button>
                </div>
                
                <div class="products-scroll-container">
                    <div class="products-scroll">
                        <?php foreach (array_slice($allProducts, 0, 8) as $product): ?>
                            <?php include 'includes/product-card.php'; ?>
                        <?php endforeach; ?>
                    </div>
                </div>
            </section>
        <?php endif; ?>

        <!-- Uncategorized Products (if any) -->
        <?php if (isset($productsByCategory['uncategorized']) && !empty($productsByCategory['uncategorized']['products'])): ?>
        <section class="product-section">
            <div class="section-header">
                <div class="section-title">
                    <h2>Other Products</h2>
                    <p class="section-subtitle">Additional Fragrances</p>
                </div>
                <button class="see-all-btn" onclick="window.location.href='product-catalog.php?category=uncategorized'">see all ></button>
            </div>
            
            <div class="products-scroll-container">
                <div class="products-scroll">
                    <?php foreach (array_slice($productsByCategory['uncategorized']['products'], 0, 4) as $product): ?>
                        <?php include 'includes/product-card.php'; ?>
                    <?php endforeach; ?>
                </div>
            </div>
        </section>
        <?php endif; ?>
    </main>

    <!-- Bottom Navigation -->
    <nav class="bottom-nav">
        <div class="nav-item" onclick="window.location.href='index.php'">
            <i class="fas fa-home"></i>
        </div>
        <div class="nav-item active">
            <i class="fas fa-box"></i>
        </div>
        <div class="nav-item" onclick="checkAuthAndRedirect('cart.php')">
            <i class="fas fa-shopping-cart"></i>
        </div>
        <div class="nav-item" onclick="checkRegisteredUserAndRedirect('royalty-rewards.php')">
            <i class="fas fa-crown"></i>
        </div>
    </nav>

    <script>
        function goBack() {
            window.history.back();
        }

        function viewProduct(productId) {
            window.location.href = `product-detail.php?id=${productId}`;
        }


        // Search functionality
        const searchInput = document.getElementById('search-input');
        if (searchInput) {
            searchInput.addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    const query = this.value.trim();
                    if (query) {
                        window.location.href = `products.php?search=${encodeURIComponent(query)}`;
                    } else {
                        window.location.href = 'products.php';
                    }
                }
            });
        }

        // Search icon click
        const searchIcon = document.querySelector('.search-icon');
        if (searchIcon) {
            searchIcon.addEventListener('click', function() {
                const searchInput = document.getElementById('search-input');
                const query = searchInput.value.trim();
                if (query) {
                    window.location.href = `products.php?search=${encodeURIComponent(query)}`;
                } else {
                    searchInput.focus();
                }
            });
        }

        // Filter by category function
        function filterByCategory(category) {
            if (category === 'bestsellers' || category === 'new') {
                // Show all products for bestsellers and new
                window.location.href = 'products.php';
            } else {
                // Filter by specific category
                window.location.href = `products.php?category=${category}`;
            }
        }

        // Clear search function
        function clearSearch() {
            window.location.href = 'products.php';
        }

        // Auto-focus search input if there's a search query
        document.addEventListener('DOMContentLoaded', function() {
            updateCartBadge();
            
            // If there's a search query, highlight it
            const searchInput = document.getElementById('search-input');
            if (searchInput && searchInput.value) {
                searchInput.select();
            }
        });
    </script>
</body>
</html>