<?php
/**
 * Database Setup Script for Perfume Store
 * Run this script to set up your database and test the connection
 */

require_once 'config/database.php';

$setup_status = [];
$errors = [];

// Test database connection
try {
    $db = getDB();
    if ($db) {
        $setup_status[] = "✅ Database connection successful";
    } else {
        $errors[] = "❌ Database connection failed";
    }
} catch (Exception $e) {
    $errors[] = "❌ Database connection error: " . $e->getMessage();
}

// Check if tables exist
if (empty($errors)) {
    try {
        $tables = ['users', 'categories', 'products', 'product_sizes', 'cart', 'orders', 'order_items', 'reviews', 'wishlist'];
        $existing_tables = [];
        
        foreach ($tables as $table) {
            $stmt = $db->prepare("SHOW TABLES LIKE ?");
            $stmt->execute([$table]);
            if ($stmt->fetch()) {
                $existing_tables[] = $table;
            }
        }
        
        if (count($existing_tables) === count($tables)) {
            $setup_status[] = "✅ All database tables exist";
        } else {
            $missing_tables = array_diff($tables, $existing_tables);
            $setup_status[] = "⚠️ Missing tables: " . implode(', ', $missing_tables);
            $setup_status[] = "📝 Please run the SQL schema from database/schema.sql";
        }
    } catch (Exception $e) {
        $errors[] = "❌ Error checking tables: " . $e->getMessage();
    }
}

// Test model classes
try {
    require_once 'models/User.php';
    require_once 'models/Product.php';
    require_once 'models/Cart.php';
    $setup_status[] = "✅ Model classes loaded successfully";
} catch (Exception $e) {
    $errors[] = "❌ Error loading model classes: " . $e->getMessage();
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Database Setup - Perfume Store</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
            background-color: #f5f5f5;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #333;
            text-align: center;
            margin-bottom: 30px;
        }
        .status-item {
            padding: 10px;
            margin: 5px 0;
            border-radius: 5px;
            font-family: monospace;
        }
        .success {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        .warning {
            background-color: #fff3cd;
            color: #856404;
            border: 1px solid #ffeaa7;
        }
        .error {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        .setup-steps {
            background-color: #e7f3ff;
            padding: 20px;
            border-radius: 5px;
            margin: 20px 0;
        }
        .setup-steps h3 {
            margin-top: 0;
            color: #0066cc;
        }
        .setup-steps ol {
            margin: 10px 0;
        }
        .setup-steps li {
            margin: 5px 0;
        }
        .code-block {
            background-color: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 5px;
            padding: 15px;
            margin: 10px 0;
            font-family: monospace;
            overflow-x: auto;
        }
        .btn {
            display: inline-block;
            padding: 10px 20px;
            background-color: #007bff;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            margin: 10px 5px;
        }
        .btn:hover {
            background-color: #0056b3;
        }
        .btn-success {
            background-color: #28a745;
        }
        .btn-success:hover {
            background-color: #1e7e34;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🗄️ Database Setup Status</h1>
        
        <?php if (!empty($setup_status)): ?>
            <h3>Setup Status:</h3>
            <?php foreach ($setup_status as $status): ?>
                <div class="status-item success"><?php echo $status; ?></div>
            <?php endforeach; ?>
        <?php endif; ?>
        
        <?php if (!empty($errors)): ?>
            <h3>Errors:</h3>
            <?php foreach ($errors as $error): ?>
                <div class="status-item error"><?php echo $error; ?></div>
            <?php endforeach; ?>
        <?php endif; ?>
        
        <div class="setup-steps">
            <h3>📋 Setup Instructions</h3>
            <ol>
                <li><strong>Create Database:</strong> Create a MySQL database named 'perfume_store'</li>
                <li><strong>Update Configuration:</strong> Edit <code>config/database.php</code> with your database credentials</li>
                <li><strong>Import Schema:</strong> Run the SQL commands from <code>database/schema.sql</code></li>
                <li><strong>Test Connection:</strong> Refresh this page to verify everything is working</li>
            </ol>
        </div>
        
        <div class="setup-steps">
            <h3>🔧 Database Configuration</h3>
            <p>Update these settings in <code>config/database.php</code>:</p>
            <div class="code-block">
define('DB_HOST', 'localhost');<br>
define('DB_NAME', 'perfume_store');<br>
define('DB_USER', 'root');<br>
define('DB_PASS', 'your_password_here');
            </div>
        </div>
        
        <div class="setup-steps">
            <h3>📊 Database Schema</h3>
            <p>Run the following SQL commands in your MySQL database:</p>
            <div class="code-block">
-- Create database<br>
CREATE DATABASE IF NOT EXISTS perfume_store;<br>
USE perfume_store;<br><br>
-- Then run all the CREATE TABLE statements from database/schema.sql
            </div>
        </div>
        
        <div class="setup-steps">
            <h3>🧪 Test Your Setup</h3>
            <p>Once the database is set up, test these features:</p>
            <ul>
                <li>✅ User registration and login</li>
                <li>✅ Product browsing and search</li>
                <li>✅ Add products to cart</li>
                <li>✅ Guest checkout</li>
            </ul>
        </div>
        
        <div style="text-align: center; margin-top: 30px;">
            <a href="index.php" class="btn btn-success">🏠 Go to Homepage</a>
            <a href="products.php" class="btn">🛍️ Browse Products</a>
            <a href="login.php" class="btn">👤 Login</a>
        </div>
        
        <?php if (empty($errors)): ?>
            <div class="status-item success" style="text-align: center; margin-top: 20px;">
                🎉 <strong>Database setup completed successfully!</strong><br>
                Your perfume store is now connected to SQL database.
            </div>
        <?php endif; ?>
    </div>
</body>
</html>