<?php
session_start();
require_once 'models/User.php';
require_once 'config/google-oauth.php';

// Clear any existing session for testing
if (isset($_GET['clear'])) {
    session_destroy();
    header('Location: signup.php');
    exit();
}

// Only redirect if already logged in and not a guest
if (isset($_SESSION['user_id']) && $_SESSION['user_id'] !== 'guest' && !isset($_GET['debug'])) {
    header('Location: index.php');
    exit();
}

$error_message = '';
$success_message = '';

// Handle OAuth errors
if (isset($_GET['error'])) {
    switch ($_GET['error']) {
        case 'oauth_cancelled':
            $error_message = 'Google sign-up was cancelled.';
            break;
        case 'oauth_failed':
            $error_message = 'Google sign-up failed. Please try again.';
            break;
    }
}

// Handle signup success
if (isset($_GET['signup']) && $_GET['signup'] === 'success') {
    $success_message = 'Successfully signed up with Google!';
}

// Handle signup required message
if (isset($_GET['message']) && $_GET['message'] === 'signup_required') {
    $error_message = 'Please sign up to continue with your purchase.';
}

// Handle signup form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $first_name = trim($_POST['first_name'] ?? '');
    $last_name = trim($_POST['last_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $password = trim($_POST['password'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $privacy_policy = isset($_POST['privacy_policy']) ? true : false;
    
    if (empty($first_name) || empty($last_name) || empty($email) || empty($password)) {
        $error_message = 'Please fill in all required fields.';
    } elseif (!$privacy_policy) {
        $error_message = 'You must agree to the Privacy Policy to create an account.';
    } elseif (strlen($password) < 6) {
        $error_message = 'Password must be at least 6 characters long.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error_message = 'Please enter a valid email address.';
    } else {
        $user = new User();
        
        // Check if email already exists
        if ($user->emailExists($email)) {
            $error_message = 'An account with this email already exists.';
        } else {
            // Create new user
            $user_data = [
                'first_name' => $first_name,
                'last_name' => $last_name,
                'email' => $email,
                'password' => $password,
                'phone' => $phone
            ];
            
            if ($user->create($user_data)) {
                // Get the created user data
                $new_user = $user->getByEmail($email);
                $_SESSION['user_id'] = $new_user['id'];
                $_SESSION['user_email'] = $new_user['email'];
                $_SESSION['user_name'] = $new_user['first_name'] . ' ' . $new_user['last_name'];
                
                // Check if there's a redirect URL stored
                $redirect_url = 'index.php';
                if (isset($_SESSION['redirect_after_signup'])) {
                    $redirect_url = $_SESSION['redirect_after_signup'];
                    unset($_SESSION['redirect_after_signup']);
                }
                
                $success_message = 'Account created successfully! Redirecting...';
                // Redirect after 2 seconds to show success message
                echo '<script>setTimeout(function(){ window.location.href = "' . htmlspecialchars($redirect_url) . '"; }, 2000);</script>';
            } else {
                $error_message = 'Failed to create account. Please try again.';
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sign Up - 2 Days Parfume</title>
    <link rel="stylesheet" href="styles.css">
    <link rel="stylesheet" href="auth.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="script.js" defer></script>
</head>
<body>
    <div class="auth-container">
        <!-- Top Navigation -->
        <nav class="top-nav">
            <div class="nav-left">
                <i class="fas fa-arrow-left" onclick="window.location.href='index.php'"></i>
            </div>
            <div class="nav-center">
                <div class="logo">
                    <img src="images/LOGO.png" alt="2 Days Parfume Logo" class="logo-img">
                </div>
            </div>
            <div class="nav-right">
                <i class="fas fa-search" onclick="window.location.href='products.php'"></i>
            </div>
        </nav>

        <!-- Auth Content -->
        <div class="auth-content">
            <div class="auth-header">
                <h1 class="auth-title">Sign Up</h1>
                <p class="auth-subtitle">Create an account so you can order your favorite products easily and quickly.</p>
            </div>

            <!-- Error/Success Messages -->
            <?php if ($error_message): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i>
                    <?php echo htmlspecialchars($error_message); ?>
                </div>
            <?php endif; ?>

            <?php if ($success_message): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i>
                    <?php echo htmlspecialchars($success_message); ?>
                </div>
            <?php endif; ?>

            <!-- Signup Form -->
            <form class="auth-form" method="POST" action="">
                <div class="form-group">
                    <input type="text" name="first_name" id="first_name" class="form-input" placeholder="First Name" required>
                </div>
                
                <div class="form-group">
                    <input type="text" name="last_name" id="last_name" class="form-input" placeholder="Last Name" required>
                </div>
                
                <div class="form-group">
                    <input type="email" name="email" id="email" class="form-input" placeholder="Email" required>
                </div>
                
                <div class="form-group">
                    <input type="password" name="password" id="password" class="form-input" placeholder="Password" required>
                    <i class="fas fa-eye password-toggle" onclick="togglePassword(this)"></i>
                </div>
                
                <div class="form-group">
                    <input type="tel" name="phone" id="phone" class="form-input" placeholder="Phone Number (Optional)">
                </div>

                <!-- Privacy Policy Checkbox -->
                <div class="form-group checkbox-group">
                    <label class="checkbox-label">
                        <input type="checkbox" name="privacy_policy" id="privacy_policy" required>
                        <span class="checkmark"></span>
                        <span class="checkbox-text">
                            I agree to the <a href="mobile-privacy-policy.php" target="_blank" class="privacy-link">Privacy Policy</a>
                        </span>
                    </label>
                </div>

                <button type="submit" class="auth-btn primary">
                    <span>Register now</span>
                </button>
            </form>

            <!-- Social Login -->
            <!-- <div class="social-divider">
                <span>Or Continue With</span>
            </div>

            <div class="social-buttons">
                <button class="social-btn google" onclick="signUpWithGoogle()">
                    <i class="fab fa-google"></i>
                    <span>Continue with Google</span>
                </button>
            </div> -->


            <!-- Login Link -->
            <div class="auth-footer">
                <p>Already Have An Account? <a href="login.php" class="auth-link">Log In</a></p>
            </div>
        </div>
    </div>

    <script>
        function togglePassword(icon) {
            const input = icon.previousElementSibling;
            if (input.type === 'password') {
                input.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                input.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }

        // Privacy Policy checkbox validation
        document.addEventListener('DOMContentLoaded', function() {
            const form = document.querySelector('.auth-form');
            const privacyCheckbox = document.getElementById('privacy_policy');
            const checkboxGroup = document.querySelector('.checkbox-group');
            
            form.addEventListener('submit', function(e) {
                if (!privacyCheckbox.checked) {
                    e.preventDefault();
                    checkboxGroup.classList.add('error');
                    
                    // Show error message
                    let errorMsg = document.querySelector('.privacy-error');
                    if (!errorMsg) {
                        errorMsg = document.createElement('div');
                        errorMsg.className = 'alert alert-error privacy-error';
                        errorMsg.innerHTML = '<i class="fas fa-exclamation-circle"></i> You must agree to the Privacy Policy to create an account.';
                        checkboxGroup.appendChild(errorMsg);
                    }
                    
                    // Scroll to checkbox
                    checkboxGroup.scrollIntoView({ behavior: 'smooth', block: 'center' });
                    return false;
                } else {
                    checkboxGroup.classList.remove('error');
                    const errorMsg = document.querySelector('.privacy-error');
                    if (errorMsg) {
                        errorMsg.remove();
                    }
                }
            });
            
            // Remove error state when checkbox is checked
            privacyCheckbox.addEventListener('change', function() {
                if (this.checked) {
                    checkboxGroup.classList.remove('error');
                    const errorMsg = document.querySelector('.privacy-error');
                    if (errorMsg) {
                        errorMsg.remove();
                    }
                }
            });
        });

        function signUpWithGoogle() {
            // Mobile-specific OAuth handling
            const isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
            const oauthUrl = '<?php echo getGoogleOAuthURL(); ?>';
            
            if (isMobile) {
                // For mobile, use a more reliable method
                try {
                    // Try opening in same window first
                    window.location.replace(oauthUrl);
                } catch (e) {
                    // Fallback to regular redirect
                    window.location.href = oauthUrl;
                }
            } else {
                // Desktop behavior
                window.location.href = oauthUrl;
            }
        }
    </script>
</body>
</html>